<?php

namespace App\Http\Controllers;

use DB;
use Carbon\Carbon;
use App\Models\User;
use App\Models\Eventos;
use App\Models\Cliente;
use App\Models\Lugares;
use App\Models\Sucursal;
use App\Models\Reservacion;
use Illuminate\Http\Request;
use App\Models\LayoutEvento;
use App\Models\MesasPosicion;
use App\Models\LayoutEventoMesa;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Spatie\Permission\Traits\HasRoles;
use Yajra\DataTables\Facades\DataTables;

class ReservacionController extends Controller
{
    use HasRoles;

    public function __construct()
    {
        $this->middleware('permission:listar_reservacion', ['only' => ['index']]);
        $this->middleware('permission:crear_reservacion', ['only' => ['create','store']]);
        $this->middleware('permission:editar_reservacion', ['only' => ['edit','update']]);
    }

    function index()
    {
        $usuarioAuth = Auth::user();
        $sucursales = Sucursal::where('status', true)->orderBy('nombre');
        if (!$usuarioAuth->can('ver_todas_sucursales'))
        {
            $sucursales->where('id', $usuarioAuth->sucursal->id);
        }
        $sucursales = $sucursales->get();

        $eventos = Eventos::orderBy('nombre')->get();

        return view('reservaciones.reservacion',[
            'eventos' => $eventos,
            'sucursales' => $sucursales,
        ]);
    }
    
    public function datatablelista_reservacion(Request $request)
    {
        //dd($request->tipo,$request->sucursal,$request->evento,$request->fecha_ini,$request->fecha_fin);

        $usuarioAuth = Auth::user();
        $reservacion_venta = Reservacion::select('reservacion_venta.id', 'eventos.nombre as evento_nombre', 'eventos.fecha', 'eventos.hora', 'sucursal.nombre as sucursal_nombre', 'reservacion_venta.tipo', 'reservacion_venta.nombre as cliente', 'reservacion_venta.telefono')
            ->join('eventos', 'eventos.id', '=', 'reservacion_venta.id_evento')
            ->join('sucursal', 'sucursal.id', '=', 'eventos.id_sucursal');

        // Sin permiso solo ve la sucursal que tiene asignada
        if (!$usuarioAuth->can('ver_todas_sucursales'))
        {
            $reservacion_venta->where('sucursal.id', $usuarioAuth->sucursal_id);
        }
        if($request->tipo != 2)
        {
            $reservacion_venta->where('reservacion_venta.tipo', $request->tipo);
        }
        if($request->sucursal != 0)
        {
            $reservacion_venta->where('sucursal.id', $request->sucursal);
        }
        if($request->evento != 0)
        {
            $reservacion_venta->where('eventos.id', $request->evento);
        }
        if (!empty($request->fecha_ini))
        {
            $reservacion_venta->where('reservacion_venta.created_at', '>=', $request->fecha_ini);
        }
        if (!empty($request->fecha_fin))
        {
            $reservacion_venta->where('reservacion_venta.created_at', '<=', $request->fecha_fin);
        }

        $reservacion_venta = $reservacion_venta->get();

        return DataTables::of($reservacion_venta)
        ->addColumn('fecha', function($reservacion_venta){
            $carbon = Carbon::parse($reservacion_venta->fecha . " " . $reservacion_venta->hora);
            return $formateada1 = $carbon->format('d-m-Y h:i:s A');
        })
        ->addColumn('opciones', function($reservacion_venta){
            $acciones = '<a class="btn btn-success" href="'.route('acceder_compra', $reservacion_venta->id).'">
                            <i class="fa-solid fa-money-bill"></i>
                        </a> ';
            if (Gate::allows('editar_reservacion'))
            {
                $acciones .= '<a class="btn btn-warning" href="'.route('editar_reservacion', $reservacion_venta->id).'">
                                    <i class="fa-solid fa-pen-to-square"></i>
                                </a> ';
            }

            if (Gate::allows('eliminar_reservacion'))
            {
                $lugares = Lugares::where('id_reservacion_venta', $reservacion_venta->id)->pluck('lugares');
                $cadenaLugares = $lugares->implode(', ');

                $mensaje="<p>Estás a punto de <b>eliminar</b> la reservación con los lugares:</p>
                <p><b>$cadenaLugares</b>,</p>
                <p>Del evento:</p>
                <p><b>$reservacion_venta->evento_nombre</b>,</p>
                <p>A nombre de:</p>
                <p><b>$reservacion_venta->cliente</b>,</p>
                <p>Contacto:</p>
                <p><b>$reservacion_venta->telefono</b></p>
                <p><b>¿Deseas continuar?</b></p>";

                $acciones .= ' <button class="btn btn-danger btn_eliminar" data-mensaje="'.$mensaje.'" data-route="'.route('eliminar_reservacion', $reservacion_venta->id).'" >
                                <i class="fa-solid fa-circle-minus"></i>
                            </button> ';
            }

            if(!$reservacion_venta->tipo)
            {
                return $acciones;
            }
        })
        ->rawColumns(['fecha', 'opciones'])
        ->toJson();
    }

    public function crear_reservacion()
    {
        $eventos = Eventos::orderBy('nombre')->get();

        return view('reservaciones.create_reservacion',[
            'editar' => false,
            'eventos' => $eventos,
        ]);
    }

    public function mesas_posicion(Request $request)
    {
        $eventos = Eventos::where('id', $request->dato)
            ->select('eventos.*')  // Seleccionar todas las columnas de la tabla "eventos"
            ->selectRaw("(
                SELECT STRING_AGG(les.nombre || '-' || 
                    CASE 
                        WHEN ec.tipo = 1 THEN 'Efectivo:$' || ec.dinero
                        WHEN ec.tipo = 2 THEN 'Blue points:' || ec.blue_points
                        WHEN ec.tipo = 3 THEN 'Mixto:$' || ec.dinero || ' / BP ' || ec.blue_points
                        ELSE 'Desconocido'
                    END, '<br>')  
                FROM evento_costo ec
                INNER JOIN layout_evento_seccion les ON les.id = ec.id_evento_seccion
                WHERE ec.id_evento = eventos.id
            ) AS costo")  // Seleccionar la subconsulta para "costo"
            ->first();

        $layout_evento_mesa = MesasPosicion::select(
            'mesas_posicion.id', 
            'mesas_posicion.id_mesa', 
            'mesa.nombre as mesa', 
            'mesas_posicion.posicion', 
            'mesas_posicion.id_seccion', 
            'layout_evento_seccion.nombre as seccion', 
            'layout_evento_seccion.codigo', 
            'layout_evento_seccion.color'
        )
        ->join('layout_evento_mesa', 'layout_evento_mesa.idlayout_evento', '=', 'mesas_posicion.idlayout_evento')
        ->join('mesa', 'mesa.id', '=', 'layout_evento_mesa.idmesa')
        ->join('layout_evento_seccion', 'layout_evento_seccion.id', '=', 'mesas_posicion.id_seccion')
        ->where('mesas_posicion.idlayout_evento', $eventos['id_layout'])
        ->get();

        $lugres_ocupados = Lugares::select('lugares.lugares', 'lugares.id_mesa_posicion')
        ->join('reservacion_venta', 'reservacion_venta.id', '=', 'lugares.id_reservacion_venta')
        ->where('reservacion_venta.id_evento', $eventos['id'])
        ->get();

        $html = view('reservaciones.mesas_posicion',[
            'evento' => $eventos,
            'dato' => $request->dato,
            'eventos' => $layout_evento_mesa,
            'lugres_ocupados' => $lugres_ocupados,
        ])->render();

        return response()->json([
            'html' => $html,
        ], 200);
    }

    function guardar_reservacion(Request $request)
    {
        $icon  = "success";
        $title = 'Guardado exitoso!';
        $text  = 'Se registro la compra correctamente';

        $this->validate($request,[
            'evento'   => 'required',
            'nombre'   => 'required|max:255',
            'telefono' => 'required|max:10',
            'folio'    => 'required|max:255',
            'lugaresArray' => 'required',
        ],
        [
            'evento.required'   => 'El evento es necesario',
            'telefono.required' => 'El número teléfonico es necesario',
            'telefono.max'      => 'El número teléfonico no puede pasar mas de 10 caracteres',
            'folio.required'    => 'El folio es necesario',
            'folio.max'         => 'El folio no puede pasar mas de 10 caracteres',
            'lugaresArray.required'  => 'Los datos son necesarios para la venta',
        ]);

        $reservacion = Reservacion::create([
            'id_evento' => $request->evento,
            'nombre'    => $request->nombre,
            'telefono'  => $request->telefono,
            'folio'     => $request->folio,
            'tipo'      => false,
        ]);

        $evento = Eventos::where('id', $request->evento)->first();
        $tipoPago = "";

        if($evento['costo_tipo_deduccion'] == "mixto")
        {
            $tipoPago = "3";
        }
        else if($evento['costo_tipo_deduccion'] == "efectivo")
        {
            $tipoPago = "2";
        }
        else if($evento['costo_tipo_deduccion'] == "blue points")
        {
            $tipoPago = "1";
        }

        $datosArray = $request->lugaresArray;
        $data = json_decode($datosArray, true);

        if ($data)
        {
            foreach ($data as $key => $sillas)
            {
                foreach ($sillas as $silla)
                {
                    Lugares::create([
                        'id_reservacion_venta' => $reservacion->id,
                        'lugares'              => $silla,
                        'id_mesa_posicion'     => $key,
                        'metodo_pago'          => $tipoPago,
                    ]);
                }
            }
        }
        else
        {
            $icon  = "error";
            $title = 'Error!';
            $text  = 'No se pudo guardar los asientos correctamente, vuelva a intentar!';
        }

        return response()->json([
            'icon'  => $icon,
            'title' => $title,
            'text'  => $text,
            'ruta'  => route('lista_reservacion'),
        ], 200);
    }

    public function editar_reservacion($id_reservacion)
    {
        $reservacion = Reservacion::select('reservacion_venta.id', 'reservacion_venta.id_evento', 'reservacion_venta.nombre', 'reservacion_venta.telefono', 'eventos.costo', 'eventos.costo_tipo_deduccion')
        ->join('eventos', 'eventos.id', '=', 'reservacion_venta.id_evento')
        ->where('reservacion_venta.id', $id_reservacion)
        ->first();

        $lugares = Lugares::where('id_reservacion_venta', $id_reservacion)->get();

        return view('reservaciones.modiificacion',[
            'lugares' => $lugares,
            'reservacion' => $reservacion,
        ]);
    }

    public function actualizar_reservacion($id_reservacion, Request $request)
    {
        $this->validate($request,[
            'id_lugar'  => 'required',
            'tipo_pago' => 'required',
        ],
        [
            'id_lugar.required'  => 'Se requiere los datos del lugar',
            'tipo_pago.required' => 'Se requiere los datos del tipo de pago',
        ]);

        $id_lugar = $request->id_lugar;
        $tipo_pago = $request->tipo_pago;

        for($i=0; $i < count($id_lugar); $i++)
        {
            $lugar = Lugares::findorfail($id_lugar[$i]);
            $lugar->metodo_pago = $tipo_pago[$i];
            $lugar->save();
        }

        return response()->json([
            'icon'   => "success",
            'title'  => 'Actialización exitosa!',
            'text'   => 'Se actualizo los datos de los lugares correctamente',
            'ruta'  => route('lista_reservacion'),
        ], 200); 
    }

    public function eliminar_reservacion($id_reservacion)
    {
        $evento = Reservacion::findOrFail($id_reservacion);
        $evento->delete();

        Lugares::where('id_reservacion_venta', $id_reservacion)->delete();

        return response()->json([
            'icon'   => "success",
            'title'  => 'Eliminado exitoso!',
            'text'   => 'Se elimino la reservación correctamente',
            'dtable' => 'dt_lista_reservacion',
        ], 200); 
    }

    public function borrar_lugar($id_lugar)
    {
        $lugar = Lugares::findOrFail($id_lugar);
        $lugar->delete();

        return response()->json([
            'fila'  => $id_lugar,
            'icon'  => "success",
            'title' => 'Eliminado exitoso!',
            'text'  => 'Se elimino el lugar correctamente',
        ], 200); 
    }

    public function acceder_compra($id_reservacion)
    {
        $reservacion = Reservacion::findOrFail($id_reservacion);
        $lugares = Lugares::where('id_reservacion_venta', $id_reservacion)->pluck('lugares');
        $cadenaLugares = $lugares->implode(', ');

        return view('reservaciones.venta',[
            'reservacion'   => $reservacion,
            'cadenaLugares' => $cadenaLugares,
        ]);
    }

    public function compra($id_reservacion, Request $request)
    {
        $evento = Reservacion::findorfail($id_reservacion);
        $evento->tipo = true;
        $evento->save();

        return response()->json([
            'icon'   => "success",
            'title'  => 'Compra exitosa!',
            'text'   => 'Se hizo la compra correctamente',
            'ruta'  => route('lista_reservacion'),
        ], 200); 
    }

    public function mesas_posiciones($id_evento, $id_cliente)
    {
        $eventos = Eventos::orderBy('nombre')->select('eventos.id', 'eventos.imagen')->selectRaw("'Evento: ' || eventos.nombre || ' - Fecha & Hora del Evento: ' || eventos.fecha || ' ' || eventos.hora AS nombrelargo")->where('id', $id_evento)->first();
        $cliente = Cliente::orderBy('nombre')->where('id', $id_cliente)->first();

        return view('reservaciones.mesas_posiciones',[
            'editar' => false,
            'cliente' => $cliente,
            'eventos_all' => $eventos,
        ]);
    }
}