<?php

namespace App\Http\Controllers;

use File;
use Carbon\Carbon;
use App\Models\User;
use App\Models\Eventos;
use App\Models\Sucursal;
use App\Models\EventoCosto;
use Illuminate\Http\Request;
use App\Models\LayoutEvento;
use App\Models\BeneficioNivel;
use App\Models\LayoutEventoSeccion;
use App\Models\EventoDescuentoNivel;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use App\Models\EventoDescuentoEspecial;
use Yajra\DataTables\Facades\DataTables;

class EventosController extends Controller
{
    private $path = 'evento/';

    public function index()
    {
        return view('eventos.eventos_list');
    }

    public function datatablelista_eventos(Request $request)
    {
        $usuarioAuth = Auth::user();
        $listaeventos = Eventos::select('id', 'nombre', 'fecha', 'hora', 'id_sucursal', 'espacio', 'id_layout', 'capacidad', 'imagen');

        // Sin permiso solo ve la sucursal que tiene asignada
        if (!$usuarioAuth->can('ver_todas_sucursales'))
        {
            $listaeventos->whereHas('Sucursal', function ($query) use ($usuarioAuth) {
                $query->where('idsucursal', $usuarioAuth->idsucursal);
            });
        }
        $listaeventos = $listaeventos->get();

        return DataTables::of($listaeventos)
        ->addColumn('imagen', function($listaeventos){
            return "<img src='".asset($listaeventos->imagen)."' class='img-fluid ver_imagen' style='cursor: pointer;'>";
        })
        ->addColumn('fecha', function($listaeventos){
            $carbon = Carbon::parse($listaeventos->fecha);
            return $formateada1 = $carbon->format('d-m-Y');
        })
        ->addColumn('hora', function($listaeventos){
            $carbon = Carbon::parse($listaeventos->hora);
            return $formateada1 = $carbon->format('h:m:s A');
        })
        ->addColumn('id_sucursal', function($listaeventos){
            $sucursales = Sucursal::findorfail($listaeventos->id_sucursal);
            return $sucursales->nombre;
        })
        ->addColumn('id_layout', function($listaeventos){
            $LayoutEvento = LayoutEvento::findorfail($listaeventos->id_layout);
            return $LayoutEvento->nombre;
        })
        ->addColumn('opciones', function($listaeventos){
            $acciones = '';
            if (Gate::allows('editar_eventos'))
            {
                $acciones .= '<a class="btn btn-warning" href="'.route('editar_eventos', $listaeventos->id).'">
                                    <i class="fa-solid fa-pen-to-square"></i>
                                </a> ';
            }

            if (Gate::allows('eliminar_eventos'))
            {
                $acciones .= ' <button class="btn btn-danger btn_eliminar" data-route="'.route('eliminar_evento', $listaeventos->id).'" >
                                <i class="fa-solid fa-circle-minus"></i>
                            </button> ';
            }

            return $acciones;
        })
        ->rawColumns(['imagen', 'fecha', 'hora', 'opciones'])
        ->toJson();
    }

    public function crear_evento(Request $request)
    {
        $usuarioAuth = Auth::user();
        $sucursales = Sucursal::where('status', true)->orderBy('nombre');
        if (!$usuarioAuth->can('ver_todas_sucursales'))
        {
            $sucursales->where('id', $usuarioAuth->sucursal->id);
        }
        $sucursales = $sucursales->get();

        $nivel = BeneficioNivel::where('status', true)->where('deleted_at', null)->get();
        $layout = LayoutEvento::where('status', true)->get();

        $zonas = LayoutEventoSeccion::select('id', 'nombre', 'idlayout_evento')->get();

        return view('eventos.eventos_create',[
            'editar' => false,
            'nivel' => $nivel,
            'zonas' => $zonas,
            'layout' => $layout,
            'sucursales' => $sucursales,
        ]);
    }

    public function guardar_evento(Request $request)
    {
        $this->validate($request,[
            'nombreevento' => 'required|max:255',
            'fechaevento'  => 'required|max:20',
            'horaevento'   => 'required|max:20',
            'idSucursal'   => 'required|numeric',
            'espacio'      => 'required|max:100',
            'idlayout'     => 'required|numeric',
            'capacidad'    => 'required|max:100',
            'maximobp'     => 'required|numeric',
            'imagen'       => 'required|mimes:png,jpg,jpeg|max:5120'
        ],
        [
            'nombreevento.required' => 'El nombre del evento es necesario',
            'nombreevento.max'      => 'El nombre no puede pasar mas de 255 caracteres',
            'fechaevento.required'  => 'La fecha del evento es necesario',
            'fechaevento.max'       => 'La fecha del evento no puede pasar mas de 20 caracteres',
            'horaevento.required'   => 'La hora del evento es necesario',
            'horaevento.max'        => 'La hora del evento no puede pasar mas de 20 caracteres',
            'idSucursal.required'   => 'La sucursal es necesario',
            'idSucursal.numeric'    => 'La sucursal debe ser numerica',
            'espacio.required'      => 'El espacio es necesario',
            'espacio.max'           => 'El espacio no puede pasar mas de 100 caracteres',
            'idlayout.required'     => 'El layout del evento es necesario',
            'idlayout.numeric'      => 'El layout del evento debe ser numerica',
            'capacidad.required'    => 'La capacidad es necesario',
            'capacidad.max'         => 'La capacidad no puede pasar mas de 100 caracteres',
            'imagen.required'       => 'La imagen es necesario',
            'imagen.mimes'          => 'La imagen debe de tener formato de png, jpg o jpeg',
            'imagen.max'            => 'La imagen no puede pasar mas de 5 mb',
            'maximobp.required'     => 'Los Boletos BP es necesario',
            'maximobp.numeric'      => 'Los Boletos BP debe ser numerica',
        ]);

        $alerta = "";

        if (!file_exists($this->path)) {
            File::makeDirectory($this->path, $mode = 0777, true, true);
        }

        if ($request->hasFile('imagen'))
        {
            $img = $request->file('imagen');
            $file_name = uniqid().".".$img->extension();
            $img->move($this->path, $file_name);
            $ruta = $this->path.$file_name;
        }

        $lastId = Eventos::max('id');
        $newId = $lastId + 1;

        $eventos = Eventos::create([
            'id'             => $newId,
            'nombre'         => $request->nombreevento,
            'fecha'          => $request->fechaevento,
            'hora'           => $request->horaevento,
            'id_sucursal'    => $request->idSucursal,
            'espacio'        => $request->espacio,
            'id_layout'      => $request->idlayout,
            'capacidad'      => $request->capacidad,
            'max_boletos_bp' => $request->maximobp,
            'imagen'         => $ruta,
        ]);

        $costo  = $request->costo;
        $moneda = $request->moneda;
        $zona   = $request->zona;

        if (is_array($costo) && is_array($moneda) && is_array($zona))
        {
            if (count($costo) !== count($moneda) || count($costo) !== count($zona))
            {
                $alerta .= "No se registraron tipos de pago";
            }
            else
            {
                for ($i = 0; $i < count($costo); $i++)
                {
                    $dinero = 0;
                    $bp = 0;
                    $tipo = null;
    
                    if ($moneda[$i] == "efectivo")
                    {
                        $dinero = $costo[$i];
                        $tipo = 1;
                    }
                    if ($moneda[$i] == "bp")
                    {
                        $bp = $costo[$i];
                        $tipo = 2;
                    }
                    if ($moneda[$i] == "mixto")
                    {
                        $partes = explode('/', $costo[$i]);

                        $dinero = $partes[0];
                        $bp = $partes[1];
                        $tipo = 3;
                    }
                    
                    $lastId = EventoCosto::max('id');
                    $newId = $lastId + 1;

                    EventoCosto::create([
                        'id'        => $newId,
                        'id_evento' => $eventos->id,
                        'id_evento_seccion' => $zona[$i],
                        'tipo' => $tipo,
                        'dinero' => $dinero,
                        'blue_points' => $bp,
                    ]);
                }
            }
        }
        else
        {
            $alerta .= "No se registraron tipos de pago";
        }

        $descuentopornivel = $request->descuentopornivel;
        $niveltipo  = $request->niveltipo;

        if (is_array($descuentopornivel) && is_array($niveltipo)) {
            if (count($descuentopornivel) !== count($niveltipo)) {
                $alerta .= "<br>No se registraron descuentos por nivel";
            } else {
                for ($i = 0; $i < count($descuentopornivel); $i++) {
                    $lastId = EventoDescuentoNivel::max('id');
                    $newId = $lastId + 1;

                    EventoDescuentoNivel::create([
                        'id'        => $newId,
                        'id_evento' => $eventos->id,
                        'id_beneficio_nivel' => $niveltipo[$i],
                        'descuento' => $descuentopornivel[$i],
                        'status' => true,
                    ]);
                }
            }
        } else {
            $alerta .= "<br>No se registraron descuentos por nivel";
        }

        $descuentoespecialTipo = $request->descuentoespecialTipo;
        $descuentoespecial1 = $request->descuentoespecial1;
        $descuentoespecial2 = $request->descuentoespecial2;
        $conceptodescuentoespecial = $request->conceptodescuentoespecial;

        if (is_array($descuentoespecialTipo) && is_array($descuentoespecial1) && is_array($descuentoespecial2) && is_array($conceptodescuentoespecial))
        {
            if (count($descuentoespecialTipo) !== count($descuentoespecial1) && count($descuentoespecialTipo) !== count($descuentoespecial2) && count($descuentoespecialTipo) !== count($conceptodescuentoespecial))
            {
                $alerta .= "<br>No se registraron descuentos especiales";
            }
            else
            {
                for ($i = 0; $i < count($descuentoespecialTipo); $i++)
                {
                    $lastId = EventoDescuentoEspecial::max('id');
                    $newId = $lastId + 1;
                    EventoDescuentoEspecial::create([
                        'id'        => $newId,
                        'id_evento' => $eventos->id,
                        'tipo' => $descuentoespecialTipo[$i],
                        'valor' => $descuentoespecial1[$i],
                        'concepto' => $conceptodescuentoespecial[$i],
                        'status' => true,
                        'boletos_gratis' => $descuentoespecial2[$i],
                    ]);
                }
            }
        }
        else
        {
            $alerta .= "<br>No se registraron descuentos especiales";
        }

        return response()->json([
            'icon'  => "success",
            'title' => 'Guardado exitoso!',
            'text'  => 'Se registro el evento correctamente. '.$alerta,
            'ruta'  => route('lista_eventos'),
        ], 200);
    }

    public function editar_eventos($id_evento)
    {
        $usuarioAuth = Auth::user();
        $sucursales = Sucursal::where('status', true)->orderBy('nombre');
        if (!$usuarioAuth->can('ver_todas_sucursales'))
        {
            $sucursales->where('id', $usuarioAuth->sucursal->id);
        }
        $sucursales = $sucursales->get();

        $nivel = BeneficioNivel::where('status', true)->where('deleted_at', null)->get();
        $layout = LayoutEvento::where('status', true)->get();

        $zonas = LayoutEventoSeccion::select('id', 'nombre', 'idlayout_evento')->get();

        //$evento = Eventos::findOrFail($id_evento);
        $evento = Eventos::select('eventos.id', 'eventos.nombre', 'eventos.fecha', 'eventos.hora', 'eventos.id_sucursal', 'eventos.espacio', 'eventos.id_layout', 'eventos.imagen', 'eventos.capacidad', 'eventos.max_boletos_bp', 'layout_evento.sillas')
        ->join('layout_evento', 'layout_evento.id', '=', 'eventos.id_layout')
        ->where('eventos.id', $id_evento)
        ->first();

        $zonasselect = LayoutEventoSeccion::select('id', 'nombre', 'idlayout_evento')
        ->where('idlayout_evento', $evento['id_layout'])
        ->get();

        $EventoCosto = EventoCosto::select('evento_costo.id', 'evento_costo.id_evento', 'evento_costo.id_evento_seccion', 'evento_costo.tipo', 'evento_costo.dinero', 'evento_costo.blue_points', 'layout_evento_seccion.nombre')
        ->join('layout_evento_seccion', 'layout_evento_seccion.id', '=', 'evento_costo.id_evento_seccion')
        ->where('evento_costo.id_evento', $id_evento)
        ->get();

        $EventoDescuentoNivel = EventoDescuentoNivel::join('beneficio_nivel', 'beneficio_nivel.id', '=', 'evento_descuento_nivel.id_beneficio_nivel')
            ->select(
                'evento_descuento_nivel.id as evento_descuento_nivel_id',
                'evento_descuento_nivel.id_evento',
                'evento_descuento_nivel.id_beneficio_nivel',
                'beneficio_nivel.nombre',
                'evento_descuento_nivel.descuento',
                'evento_descuento_nivel.status'
            )
            ->where('evento_descuento_nivel.id_evento', $id_evento)
            ->get();

        $EventoDescuentoEspecial = EventoDescuentoEspecial::where('id_evento', $id_evento)->get();
        
        return view('eventos.eventos_create',[
            'editar' => true,
            'zonas' => $zonas,
            'zonasselect' => $zonasselect,
            'nivel' => $nivel,
            'layout' => $layout,
            'evento' => $evento,
            'sucursales' => $sucursales,
            'eventocosto' => $EventoCosto,
            'eventodescuentonivel' => $EventoDescuentoNivel,
            'eventodescuentoespecial' => $EventoDescuentoEspecial,
        ]);
    }

    public function actualizar_eventos(Request $request, $id_evento)
    {   
        $this->validate($request,[
            'nombreevento' => 'required|max:255',
            'fechaevento'  => 'required|max:20',
            'horaevento'   => 'required|max:20',
            'idSucursal'   => 'required|numeric',
            'espacio'      => 'required|max:100',
            'idlayout'     => 'required|numeric',
            'capacidad'    => 'required|max:100',
            'maximobp'     => 'required|numeric',
            'imagen'       => 'nullable|mimes:png,jpg,jpeg|max:5120'
        ],
        [
            'nombreevento.required' => 'El nombre del evento es necesario',
            'nombreevento.max'      => 'El nombre no puede pasar mas de 255 caracteres',
            'fechaevento.required'  => 'La fecha del evento es necesario',
            'fechaevento.max'       => 'La fecha del evento no puede pasar mas de 20 caracteres',
            'horaevento.required'   => 'La hora del evento es necesario',
            'horaevento.max'        => 'La hora del evento no puede pasar mas de 20 caracteres',
            'idSucursal.required'   => 'La sucursal es necesario',
            'idSucursal.numeric'    => 'La sucursal debe ser numerica',
            'espacio.required'      => 'El espacio es necesario',
            'espacio.max'           => 'El espacio no puede pasar mas de 100 caracteres',
            'idlayout.required'     => 'El layout del evento es necesario',
            'idlayout.numeric'      => 'El layout del evento debe ser numerica',
            'capacidad.required'    => 'La capacidad es necesario',
            'capacidad.max'         => 'La capacidad no puede pasar mas de 100 caracteres',
            'imagen.required'       => 'La imagen es necesario',
            'imagen.mimes'          => 'La imagen debe de tener formato de png, jpg o jpeg',
            'imagen.max'            => 'La imagen no puede pasar mas de 5 mb',
            'maximobp.required'     => 'Los Boletos BP es necesario',
            'maximobp.numeric'      => 'Los Boletos BP debe ser numerica',
        ]);

        $eventos = Eventos::findorfail($id_evento);

        if ($request->hasFile('imagen')) {
            $img = $request->file('imagen');
            $file_name = uniqid().".".$img->extension();
            $img->move($this->path, $file_name);
            $ruta = $this->path . $file_name;

            // Eliminar la imagen anterior si existe
            if (File::exists($eventos->imagen)) {
                File::delete($eventos->imagen);
            }

            $eventos->imagen = $ruta;
        }

        $eventos->nombre         = $request->nombreevento;
        $eventos->fecha          = $request->fechaevento;
        $eventos->hora           = $request->horaevento;
        $eventos->id_sucursal    = $request->idSucursal;
        $eventos->espacio        = $request->espacio;
        $eventos->id_layout      = $request->idlayout;
        $eventos->capacidad      = $request->capacidad;
        $eventos->max_boletos_bp = $request->maximobp;
        $eventos->save();

        EventoCosto::where('id_evento', $eventos->id)->delete();
        EventoDescuentoNivel::where('id_evento', $eventos->id)->delete();
        EventoDescuentoEspecial::where('id_evento', $eventos->id)->delete();

        $costo  = $request->costo;
        $moneda = $request->moneda;
        $zona   = $request->zona;

        if (is_array($costo) && is_array($moneda) && is_array($zona))
        {
            if (count($costo) !== count($moneda) || count($costo) !== count($zona))
            {
                $alerta .= "No se registraron tipos de pago";
            }
            else
            {
                for ($i = 0; $i < count($costo); $i++)
                {
                    $dinero = 0;
                    $bp = 0;
                    $tipo = null;
    
                    if ($moneda[$i] == "efectivo")
                    {
                        $dinero = $costo[$i];
                        $tipo = 1;
                    }
                    if ($moneda[$i] == "bp")
                    {
                        $bp = $costo[$i];
                        $tipo = 2;
                    }
                    if ($moneda[$i] == "mixto")
                    {
                        $partes = explode('/', $costo[$i]);

                        $dinero = $partes[0];
                        $bp = $partes[1];
                        $tipo = 3;
                    }
                    
                    $lastId = EventoCosto::max('id');
                    $newId = $lastId + 1;

                    EventoCosto::create([
                        'id'        => $newId,
                        'id_evento' => $eventos->id,
                        'id_evento_seccion' => $zona[$i],
                        'tipo' => $tipo,
                        'dinero' => $dinero,
                        'blue_points' => $bp,
                    ]);
                }
            }
        }
        else
        {
            $alerta .= "No se registraron tipos de pago";
        }

        $descuentopornivel = $request->descuentopornivel;
        $niveltipo  = $request->niveltipo;

        if (is_array($descuentopornivel) && is_array($niveltipo)) {
            if (count($descuentopornivel) !== count($niveltipo)) {
                $alerta .= "<br>No se registraron descuentos por nivel";
            } else {
                for ($i = 0; $i < count($descuentopornivel); $i++) {
                    $lastId = EventoDescuentoNivel::max('id');
                    $newId = $lastId + 1;

                    EventoDescuentoNivel::create([
                        'id'        => $newId,
                        'id_evento' => $eventos->id,
                        'id_beneficio_nivel' => $niveltipo[$i],
                        'descuento' => $descuentopornivel[$i],
                        'status' => true,
                    ]);
                }
            }
        } else {
            $alerta .= "<br>No se registraron descuentos por nivel";
        }

        $descuentoespecialTipo = $request->descuentoespecialTipo;
        $descuentoespecial1 = $request->descuentoespecial1;
        $descuentoespecial2 = $request->descuentoespecial2;
        $conceptodescuentoespecial = $request->conceptodescuentoespecial;

        if (is_array($descuentoespecialTipo) && is_array($descuentoespecial1) && is_array($descuentoespecial2) && is_array($conceptodescuentoespecial))
        {
            if (count($descuentoespecialTipo) !== count($descuentoespecial1) && count($descuentoespecialTipo) !== count($descuentoespecial2) && count($descuentoespecialTipo) !== count($conceptodescuentoespecial))
            {
                $alerta .= "<br>No se registraron descuentos especiales";
            }
            else
            {
                for ($i = 0; $i < count($descuentoespecialTipo); $i++)
                {
                    $lastId = EventoDescuentoEspecial::max('id');
                    $newId = $lastId + 1;
                    EventoDescuentoEspecial::create([
                        'id'        => $newId,
                        'id_evento' => $eventos->id,
                        'tipo' => $descuentoespecialTipo[$i],
                        'valor' => $descuentoespecial1[$i],
                        'concepto' => $conceptodescuentoespecial[$i],
                        'status' => true,
                        'boletos_gratis' => $descuentoespecial2[$i],
                    ]);
                }
            }
        }
        else
        {
            $alerta .= "<br>No se registraron descuentos especiales";
        }

        return response()->json([
            'icon'  => "success",
            'title' => 'Actializado exitoso!',
            'text'  => 'Se aztualizo el evento correctamente',
            'ruta'  => route('lista_eventos'),
        ], 200);
    }

    public function eliminar_eventos($id_evento)
    {
        $eventos = Eventos::findOrFail($id_evento);
        File::delete($eventos->imagen);
        $eventos->delete();

        EventoCosto::where('id_evento', $eventos->id)->delete();
        EventoDescuentoNivel::where('id_evento', $eventos->id)->delete();
        EventoDescuentoEspecial::where('id_evento', $eventos->id)->delete();

        return response()->json([
            'icon'   => "success",
            'title'  => 'Eliminado exitoso!',
            'text'   => 'Se elimino el evento correctamente',
            'dtable' => 'dt_lista_eventos',
        ], 200); 
    }

    public function eliminar_evento_opcion($id, $opcion, $div)
    {
        if($opcion === "1")
        {
            $metodo_pago = EventoCosto::findOrFail($id);
            $metodo_pago->delete();
        }
        else if($opcion === "2")
        {
            $descuentoNivel = EventoDescuentoNivel::findOrFail($id);
            $descuentoNivel->delete();
        }
        else if($opcion === "3")
        {
            $descuentoEsp = EventoDescuentoEspecial::findOrFail($id);
            $descuentoEsp->delete();
        }

        return response()->json([
            'icon'   => "success",
            'title'  => 'Eliminado exitoso!',
            'text'   => 'Se elimino correctamente',
            'estatus' => false,
            'id_div' => $div,
        ], 200); 
    }

    public function status_evento_opcion($id, $opcion, $div)
    {
        $boton = "";
        $texto = "";
        $texto2 = "";
        
        if($opcion === "1")
        {
            $metodo_pago = EventoCosto::findOrFail($id);

            if($metodo_pago->status)
            {
                $boton = "warning";
                $texto = "Desactivado";
                $texto2 = "¿Activar este metodo de pago?";
            }
            else
            {
                $boton = "success";
                $texto = "Activado";
                $texto2 = "¿Desactivar este metodo de pago?";
            }

            $metodo_pago->status = !$metodo_pago->status;
            $metodo_pago->save();
        }
        else if($opcion === "2")
        {
            //$descuentoNivel = EventoDescuentoNivel::findOrFail($id);
            
            $descuentoNivel = EventoDescuentoNivel::join('beneficio_nivel', 'beneficio_nivel.id', '=', 'evento_descuento_nivel.id_beneficio_nivel')
            ->select('evento_descuento_nivel.*', 'beneficio_nivel.nombre')
            ->where('evento_descuento_nivel.id', $id)
            ->first();

            if($descuentoNivel->status)
            {
                $boton = "warning";
                $texto = "Desactivado";
                $texto2 = "¿Activar este descuento del nivel {$descuentoNivel->nombre}?";
            }
            else
            {
                $boton = "success";
                $texto = "Activado";
                $texto2 = "¿Desactivar este descuento del nivel {$descuentoNivel->nombre}?";
            }

            $descuentoNivel->status = !$descuentoNivel->status;
            $descuentoNivel->save();
        }
        else if($opcion === "3")
        {
            $descuentoEsp = EventoDescuentoEspecial::findOrFail($id);

            if($descuentoEsp->status)
            {
                $boton = "warning";
                $texto = "Desactivado";
                $texto2 = "¿Activar este descuento del nivel {$descuentoEsp->concepto}?";
            }
            else
            {
                $boton = "success";
                $texto = "Activado";
                $texto2 = "¿Desactivar este descuento del nivel {$descuentoEsp->concepto}?";
            }

            $descuentoEsp->status = !$descuentoEsp->status;
            $descuentoEsp->save();
        }

        return response()->json([
            'icon'   => "success",
            'text'   => 'Se cambio el estatus correctamente',
            'estatus' => true,
            'textoestatus' => true,
            'id_div' => $div,
            'boton' => "btn-".$boton." btn-sm btn_tabla",
            'texto' => $texto,
            'texto2' => $texto2,
        ], 200); 
    }
}
