<?php

namespace App\Http\Controllers;

use File;
use Carbon\Carbon;
use App\Models\User;
use App\Models\Mesa;
use App\Models\Sucursal;
use App\Models\MesaSucursal;
use Illuminate\Http\Request;
use App\Models\LayoutEventoMesa;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Spatie\Permission\Traits\HasRoles;
use Yajra\DataTables\Facades\DataTables;

class CatalogoMesasController extends Controller
{
    use HasRoles;
    private $path = 'icono_mesas/';

    public function __construct()
    {
        $this->middleware('permission:listar_catalogo_mesas', ['only' => ['index']]);
        $this->middleware('permission:crear_catalogo_mesas', ['only' => ['create','store']]);
        $this->middleware('permission:editar_catalogo_mesas', ['only' => ['edit','update']]);
    }

    function index()
    {
        return view('catalogomesas.catalogomesas');
    }
    
    public function datatablelista_mesas(Request $request)
    {
        $usuarioAuth = Auth::user();
        $catalogomesas = Mesa::select('id', 'nombre', 'sillas', 'activo', 'icono');

        // Sin permiso solo ve la sucursal que tiene asignada
        if (!$usuarioAuth->can('ver_todas_sucursales'))
        {
            $catalogomesas->whereHas('Sucursal', function ($query) use ($usuarioAuth) {
                $query->where('id_sucursal', $usuarioAuth->idsucursal);
            });
        }
        $catalogomesas = $catalogomesas->get();

        return DataTables::of($catalogomesas)
        ->addColumn('icono', function($catalogomesas){
            return "<img src='".asset($catalogomesas->icono)."' class='img-fluid' alt='icono'>";
        })
        ->addColumn('sucursal', function($catalogomesas){
            $sucursal = MesaSucursal::join('mesa', 'mesa.id', '=', 'mesa_sucursal.idmesa')
                        ->join('sucursal', 'sucursal.id', '=', 'mesa_sucursal.idsucursal')
                        ->select('sucursal.nombre')
                        ->where('mesa.id', $catalogomesas->id)
                        ->get();

            $lista = "<ul>";
            foreach ($sucursal as $result)
            {
                $lista .= "<li>$result->nombre</li>";
            }
            return $lista .= "</ul>";
        })
        ->addColumn('activo', function($catalogomesas){
            return $catalogomesas->activo ? "Activo" : "Desactivado";
        })
        ->addColumn('opciones', function($catalogomesas){
            $acciones = '';
            if (Gate::allows('editar_catalogo_mesas'))
            {
                $acciones .= '<a class="btn btn-warning" href="'.route('editar_mesas', $catalogomesas->id).'">
                                <i class="fa-solid fa-pen-to-square"></i>
                            </a> ';
            }

            if (Gate::allows('eliminar_catalogo_mesas'))
            {
                $acciones .= ' <button class="btn btn-danger btn_eliminar" data-route="'.route('eliminar_mesa', $catalogomesas->id).'" >
                                <i class="fa-solid fa-circle-minus"></i>
                            </button> ';
            }

            return $acciones;
        })
        ->rawColumns(['icono', 'sucursal', 'activo', 'opciones'])
        ->toJson();
    }

    public function crear_mesa()
    {
        $usuarioAuth = Auth::user();
        $sucursales = Sucursal::where('status', true)->orderBy('nombre');
        if (!$usuarioAuth->can('ver_todas_sucursales'))
        {
            $sucursales->where('id', $usuarioAuth->sucursal->id);
        }
        $sucursales = $sucursales->get();

        return view('catalogomesas.catalogomesascreate',[
            'editar' => false,
            'sucursales' => $sucursales,
        ]);
    }

    function guardar_mesa(Request $request)
    {
        $this->validate($request,[
            'nombre'   => 'required|max:255',
            'cantidad' => 'required',
            'sucursal' => 'required|array|min:1',
            'activo'   => 'required|max:1',
            'icono'    => 'required|image|mimes:jpg,png|max:5120',
        ],
        [
            'nombre.required'   => 'El nombre es necesario',
            'nombre.max'        => 'El nombre no puede pasar mas de 255 caracteres',
            'cantidad.required' => 'El cantidad de asientos es necesario',
            'sucursal.required' => 'Alerta: Se necesita seleccioner una sucursal',
            'sucursal.array'    => 'El campo sucursal debe ser un array',
            'sucursal.min'      => 'Los datos de las sucursal no puede estar vacío',
            'activo.required'   => 'El estatus es necesario',
            'activo.max'        => 'El estatus no puede pasar mas de 1 caracteres',
            'icono.required'    => 'La imagen es requerida.',
            'icono.image'       => 'El archivo debe ser una imagen.',
            'icono.mimes'       => 'La imagen debe estar en formato JPG o PNG.',
            'icono.max'         => 'La imagen no debe superar los 5 MB.',
        ]);

        $sucursal = $request->sucursal;

        if (!file_exists($this->path)) {
            File::makeDirectory($this->path, $mode = 0777, true, true);
        }

        if ($request->hasFile('icono'))
        {
            $img = $request->file('icono');
            $file_name = uniqid().".".$img->extension();
            $img->move($this->path, $file_name);
            $ruta = $this->path.$file_name;
        }

        $mesaInsert = Mesa::create([
            'nombre' => $request->nombre,
            'sillas' => $request->cantidad,
            'activo' => $request->activo,
            'icono' => $ruta,
        ]);

        for($i=0; $i<count($sucursal); $i++)
        {
            MesaSucursal::create([
                'idmesa'     => $mesaInsert['id'],
                'idsucursal' => $sucursal[$i],
            ]);
        }

        return response()->json([
            'icon'  => "success",
            'title' => 'Guardado exitoso!',
            'text'  => 'Se registro la mesa correctamente',
            'ruta'  => route('lista_mesas'),
        ], 200);
    }

    public function editar_mesas($id)
    {
        $usuarioAuth = Auth::user();
        $mesas = Mesa::findOrFail($id);
        $sucursal = MesaSucursal::where('idmesa', $id)->get();

        $sucursalIds = $sucursal->pluck('idsucursal')->toArray();

        $sucursales = Sucursal::where('status', true)->orderBy('nombre');
        if (!$usuarioAuth->can('ver_todas_sucursales'))
        {
            $sucursales->where('id', $usuarioAuth->sucursal->id);
        }
        $sucursales = $sucursales->get();

        return view('catalogomesas.catalogomesascreate', [
            'editar' => true,
            'mesas' => $mesas,
            'sucursalIds' => $sucursalIds,
            'sucursales' => $sucursales,
        ]);
    }

    function actualizar_mesas(Request $request, $id_mesa)
    {
        $this->validate($request,[
            'nombre'   => 'required|max:255',
            'cantidad' => 'required',
            'sucursal' => 'required',
            'activo'   => 'required|max:1',
            'icono'    => 'nullable|image|mimes:jpg,png|max:5120',
        ],
        [
            'nombre.required'   => 'El nombre es necesario',
            'nombre.max'        => 'El nombre no puede pasar mas de 255 caracteres',
            'cantidad.required' => 'El cantidad de asientos es necesario',
            'sucursal.required' => 'Alerta: Se necesita seleccioner una sucursal',
            'activo.required'   => 'El estatus es necesario',
            'activo.max'        => 'El estatus no puede pasar mas de 1 caracteres',
            'icono.image'       => 'El archivo debe ser una imagen.',
            'icono.mimes'       => 'La imagen debe estar en formato JPG o PNG.',
            'icono.max'         => 'La imagen no debe superar los 5 MB.',
        ]);

        $mesa = Mesa::findOrFail($id_mesa);

        if ($request->hasFile('icono')) {
            $img = $request->file('icono');
            $file_name = uniqid().".".$img->extension();
            $img->move($this->path, $file_name);
            $ruta = $this->path . $file_name;

            // Eliminar la imagen anterior si existe
            if (File::exists($mesa->imagen)) {
                File::delete($mesa->imagen);
            }

            $mesa->icono = $ruta;
        }

        $mesa->nombre = $request->nombre;
        $mesa->sillas = $request->cantidad;
        $mesa->activo = $request->activo;
        $mesa->save();

        MesaSucursal::where('idmesa', $id_mesa)->delete();
        $sucursal = $request->sucursal;
        for($i=0; $i<count($sucursal); $i++)
        {
            MesaSucursal::create([
                'idmesa'     => $id_mesa,
                'idsucursal' => $sucursal[$i],
            ]);
        }

        return response()->json([
            'icon'  => "success",
            'title' => 'Editado exitoso!',
            'text'  => 'Se actializo la mesa correctamente',
            'ruta'  => route('lista_mesas'),
        ], 200);
    }

    public function eliminar_mesas($id_mesa)
    {
        $mesaEncontrada = LayoutEventoMesa::where('idmesa', $id_mesa)->get();

        if ($mesaEncontrada->isEmpty())
        {
            $mesa = Mesa::findOrFail($id_mesa);
            File::delete($mesa->icono);
            $mesa->delete();
            MesaSucursal::where('idmesa', $id_mesa)->delete();
    
            return response()->json([
                'icon'   => "success",
                'title'  => 'Eliminado exitoso!',
                'text'   => 'Se elimino la mesa correctamente',
                'dtable' => 'dt_lista_mesas',
            ], 200); 
        }
        else
        {
            return response()->json([
                'icon'  => "warning",
                'title' => 'Eliminación detenida',
                'text'  => 'Se detectaron mesas asignadas a un layout, por lo cual no se puede borrar porque ya están asignadas. Si desea eliminar, primero debe quitar la mesa del layout.',
            ], 200); 
        }
    }
}
